const CACHE_NAME = 'arbitrage-cache-v3';
const urlsToCache = [
  '/',
  '/index.html',
  '/dashboard.html',
  '/history.html', // NEW: Offline history page
  '/manifest.json',
  '/service-worker.js',

  // === API Data Caches ===
  '/api/profits.json',           // Historical profits & ROI%
  '/api/get_rates.php',          // Live rates
  '/api/simulate.php',           // Simulation endpoint
  '/api/trades.php',             // Trade history
  '/api/ai_predictions.json',    // AI prediction data (NEW)
  '/api/update_profits.php',     // Background sync trigger (NEW)

  // === Icons / PWA ===
  '/icons/icon-192.png',
  '/icons/icon-512.png',
  '/icons/maskable-icon-192.png',
  '/icons/maskable-icon-512.png',

  // === External CDN ===
  'https://cdn.tailwindcss.com',
  'https://cdn.jsdelivr.net/npm/chart.js',
  'https://cdn.jsdelivr.net/npm/file-saver@2.0.5/dist/FileSaver.min.js', // CSV download
  'https://cdnjs.cloudflare.com/ajax/libs/jspdf/2.5.1/jspdf.umd.min.js'  // PDF export
];

// 📥 Install Service Worker & Precache Assets
self.addEventListener('install', event => {
  event.waitUntil(
    caches.open(CACHE_NAME).then(cache => cache.addAll(urlsToCache))
  );
  self.skipWaiting();
});

// ♻️ Activate SW & Clear Old Caches
self.addEventListener('activate', event => {
  event.waitUntil(
    caches.keys().then(keys =>
      Promise.all(
        keys.map(key => {
          if (key !== CACHE_NAME) {
            console.log('[SW] Removing old cache:', key);
            return caches.delete(key);
          }
        })
      )
    )
  );
  self.clients.claim();
});

// 🌐 Fetch Strategy
self.addEventListener('fetch', event => {
  const requestUrl = new URL(event.request.url);

  // Network-first for API calls to keep data fresh
  if (requestUrl.pathname.startsWith('/api/')) {
    event.respondWith(
      fetch(event.request)
        .then(response => {
          const clone = response.clone();
          caches.open(CACHE_NAME).then(cache => cache.put(event.request, clone));
          return response;
        })
        .catch(() => caches.match(event.request))
    );
    return;
  }

  // Cache-first for static assets
  event.respondWith(
    caches.match(event.request).then(response =>
      response || fetch(event.request).catch(() => caches.match('/index.html'))
    )
  );
});

// 📊 Background Sync for Profits & AI Data
self.addEventListener('sync', event => {
  if (event.tag === 'sync-profits') {
    event.waitUntil(
      fetch('/api/update_profits.php')
        .then(res => res.json())
        .then(data => {
          console.log('[SW] Profits & ROI synced:', data);
        })
        .catch(err => console.error('[SW] Profit sync failed:', err))
    );
  }

  if (event.tag === 'sync-ai-predictions') {
    event.waitUntil(
      fetch('/api/ai_predictions.json')
        .then(res => res.json())
        .then(data => {
          console.log('[SW] AI Predictions synced:', data);
        })
        .catch(err => console.error('[SW] AI sync failed:', err))
    );
  }
});

// 📩 Push Notifications (NEW)
self.addEventListener('push', event => {
  const data = event.data ? event.data.json() : {};
  const title = data.title || 'Arbitrage Update';
  const options = {
    body: data.body || 'New profit/ROI data available.',
    icon: '/icons/icon-192.png',
    badge: '/icons/icon-192.png'
  };
  event.waitUntil(self.registration.showNotification(title, options));
});
