<?php
header("Content-Type: application/json");

// ----------------------------
// 📦 Exchange Fetch Functions
// ----------------------------

// BINANCE
function fetch_binance_rate($type) {
    $url = "https://p2p.binance.com/bapi/c2c/v2/friendly/c2c/adv/search";
    $payload = [
        "asset"     => "USDT",
        "fiat"      => "NGN",
        "tradeType" => $type,
        "page"      => 1,
        "rows"      => 5
    ];
    return fetch_post_json_prices($url, $payload, "data", fn($ad) => $ad["adv"]["price"]);
}

// KUCOIN
function fetch_kucoin_rate($type) {
    $url = "https://www.kucoin.com/_api/c2c/market/oneOrderList";
    $payload = [
        "currency"    => "USDT",
        "fiat"        => "NGN",
        "side"        => strtolower($type),
        "payment"     => "",
        "currentPage" => 1,
        "pageSize"    => 5
    ];
    return fetch_post_form_prices($url, $payload, "data.items", fn($item) => $item["price"]);
}

// BYBIT
function fetch_bybit_rate($type) {
    $url = "https://api2.bybit.com/fiat/otc/item/online";
    $payload = [
        "userId"     => "",
        "tokenId"    => "USDT",
        "currencyId" => "NGN",
        "payment"    => [],
        "side"       => strtoupper($type),
        "size"       => "5",
        "page"       => "1",
        "amount"     => "",
        "authMaker"  => false,
        "canTrade"   => true
    ];
    return fetch_post_json_prices($url, $payload, "result.items", fn($item) => $item["price"]);
}

// PAXFUL
function fetch_paxful_rate($type) {
    $url = "https://paxful.com/api/offer?crypto=usdt&currency=ngn&type=" . strtolower($type);
    return fetch_get_prices($url, "data.list", fn($item) => $item["fiat_price_per_crypto"]);
}

// GATE.IO
function fetch_gateio_rate($type) {
    $url = "https://www.gate.io/json_svr/query_push_p2p_ads";
    $payload = [
        "coin"        => "USDT",
        "curr"        => "NGN",
        "type"        => strtolower($type),
        "amount"      => "",
        "country"     => "",
        "isMerchant"  => "",
        "payMethod"   => "",
        "acceptOrder" => ""
    ];
    return fetch_post_form_prices($url, $payload, "data.list", fn($item) => $item["price"]);
}

// ----------------------------
// 🔧 Utility Functions
// ----------------------------

function fetch_post_json_prices($url, $payload, $dataPath, $mapFn) {
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST           => true,
        CURLOPT_POSTFIELDS     => json_encode($payload),
        CURLOPT_HTTPHEADER     => ["Content-Type: application/json"],
        CURLOPT_TIMEOUT        => 10
    ]);
    $response = curl_exec($ch);
    if (curl_errno($ch)) return [];
    curl_close($ch);
    $json = json_decode($response, true);
    $data = extract_deep($json, $dataPath);
    if (!$data) return [];
    return array_map(fn($item) => floatval($mapFn($item)), $data);
}

function fetch_post_form_prices($url, $payload, $dataPath, $mapFn) {
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST           => true,
        CURLOPT_POSTFIELDS     => http_build_query($payload),
        CURLOPT_HTTPHEADER     => ["Content-Type: application/x-www-form-urlencoded"],
        CURLOPT_TIMEOUT        => 10
    ]);
    $response = curl_exec($ch);
    if (curl_errno($ch)) return [];
    curl_close($ch);
    $json = json_decode($response, true);
    $data = extract_deep($json, $dataPath);
    if (!$data) return [];
    return array_map(fn($item) => floatval($mapFn($item)), $data);
}

function fetch_get_prices($url, $dataPath, $mapFn) {
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT        => 10
    ]);
    $response = curl_exec($ch);
    if (curl_errno($ch)) return [];
    curl_close($ch);
    $json = json_decode($response, true);
    $data = extract_deep($json, $dataPath);
    if (!$data) return [];
    return array_map(fn($item) => floatval($mapFn($item)), $data);
}

function extract_deep($array, $path) {
    $parts = explode('.', $path);
    foreach ($parts as $part) {
        if (!isset($array[$part])) return null;
        $array = $array[$part];
    }
    return $array;
}

// ----------------------------
// 🔍 Find Best Arbitrage Rates
// ----------------------------

$exchanges = [
    "Binance" => [fetch_binance_rate("BUY"), fetch_binance_rate("SELL")],
    "KuCoin"  => [fetch_kucoin_rate("BUY"),  fetch_kucoin_rate("SELL")],
    "Bybit"   => [fetch_bybit_rate("BUY"),   fetch_bybit_rate("SELL")],
    "Paxful"  => [fetch_paxful_rate("BUY"),  fetch_paxful_rate("SELL")],
    "Gate.io" => [fetch_gateio_rate("BUY"),  fetch_gateio_rate("SELL")]
];

$bestBuyRate  = PHP_FLOAT_MAX;
$bestBuyEx    = null;
$bestSellRate = 0;
$bestSellEx   = null;

foreach ($exchanges as $name => [$buyRates, $sellRates]) {
    if (!empty($buyRates)) {
        $minBuy = min($buyRates);
        if ($minBuy < $bestBuyRate) {
            $bestBuyRate = $minBuy;
            $bestBuyEx = $name;
        }
    }
    if (!empty($sellRates)) {
        $maxSell = max($sellRates);
        if ($maxSell > $bestSellRate) {
            $bestSellRate = $maxSell;
            $bestSellEx = $name;
        }
    }
}

// ----------------------------
// 💹 Save Profit to JSON Log
// ----------------------------

$profit = max(0, ($bestSellRate - $bestBuyRate) * 100); // Profit for 100 USDT
$timestamp = date("Y-m-d H:i:s");

$logFile = __DIR__ . '/profits.json';
$newEntry = [
    "timestamp" => $timestamp,
    "buy"       => round($bestBuyRate, 2),
    "sell"      => round($bestSellRate, 2),
    "exchange"  => $bestBuyEx . " ➝ " . $bestSellEx,
    "profit"    => round($profit, 2)
];

$existingData = file_exists($logFile) ? json_decode(file_get_contents($logFile), true) : [];
$existingData[] = $newEntry;
if (count($existingData) > 100) array_shift($existingData);

file_put_contents($logFile, json_encode($existingData, JSON_PRETTY_PRINT));

// ----------------------------
// 📤 Output JSON Response
// ----------------------------

echo json_encode([
    "buy" => [
        "exchange" => $bestBuyEx,
        "rate"     => round($bestBuyRate, 2)
    ],
    "sell" => [
        "exchange" => $bestSellEx,
        "rate"     => round($bestSellRate, 2)
    ],
    "profit" => round($profit, 2)
]);
?>
