<?php
// includes/secrets.php
declare(strict_types=1);

function get_master_key(): string {
    // Prefer env var first:
    $key = getenv('ARBITRAGE_MASTER_KEY') ?: null;
    if ($key) return $key;

    // Otherwise load from file OUTSIDE webroot:
    $path = __DIR__ . '/../secrets/master.key'; // ensure this is outside webroot if possible
    if (!file_exists($path)) throw new RuntimeException('Master key missing');
    return trim(file_get_contents($path));
}

/**
 * Encrypt plaintext using AES-256-GCM; returns base64 of iv|tag|ciphertext
 */
function secret_encrypt(string $plaintext): string {
    $key = get_master_key();
    $iv = random_bytes(12); // 96-bit recommended for GCM
    $tag = '';
    $cipher = openssl_encrypt($plaintext, 'aes-256-gcm', $key, OPENSSL_RAW_DATA, $iv, $tag);
    if ($cipher === false) throw new RuntimeException('Encryption failed');
    return base64_encode($iv . $tag . $cipher);
}

/**
 * Decrypt value produced by secret_encrypt
 */
function secret_decrypt(string $b64): string {
    $data = base64_decode($b64, true);
    if ($data === false || strlen($data) < 28) throw new RuntimeException('Invalid encrypted data');
    $iv = substr($data, 0, 12);
    $tag = substr($data, 12, 16);
    $ciphertext = substr($data, 28);
    $key = get_master_key();
    $plain = openssl_decrypt($ciphertext, 'aes-256-gcm', $key, OPENSSL_RAW_DATA, $iv, $tag);
    if ($plain === false) throw new RuntimeException('Decryption failed');
    return $plain;
}
