<?php
require_once __DIR__ . '/db.php';
require_once __DIR__ . '/secrets.php';

function save_exchange_credentials(PDO $pdo, string $exchange, array $data): bool {
    $api_key = isset($data['api_key']) ? secret_encrypt($data['api_key']) : null;
    $api_secret = isset($data['api_secret']) ? secret_encrypt($data['api_secret']) : null;
    $passphrase = isset($data['passphrase']) ? secret_encrypt($data['passphrase']) : null;
    $public_key = $data['public_key'] ?? null;
    $notes = $data['notes'] ?? null;
    $enabled = isset($data['enabled']) ? (int)$data['enabled'] : 1;

    $stmt = $pdo->prepare("
        INSERT INTO exchange_credentials (exchange, api_key, api_secret, passphrase, public_key, notes, enabled)
        VALUES (?, ?, ?, ?, ?, ?, ?)
        ON DUPLICATE KEY UPDATE api_key=VALUES(api_key), api_secret=VALUES(api_secret),
            passphrase=VALUES(passphrase), public_key=VALUES(public_key), notes=VALUES(notes), enabled=VALUES(enabled)
    ");
    return $stmt->execute([$exchange, $api_key, $api_secret, $passphrase, $public_key, $notes, $enabled]);
}

function get_exchange_credentials(PDO $pdo, string $exchange): ?array {
    $stmt = $pdo->prepare("SELECT * FROM exchange_credentials WHERE exchange = ? LIMIT 1");
    $stmt->execute([$exchange]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$row) return null;
    // decrypt fields safely (wrap in try)
    try {
        $row['api_key_plain'] = $row['api_key'] ? secret_decrypt($row['api_key']) : null;
        $row['api_secret_plain'] = $row['api_secret'] ? secret_decrypt($row['api_secret']) : null;
        $row['passphrase_plain'] = $row['passphrase'] ? secret_decrypt($row['passphrase']) : null;
    } catch (Exception $e) {
        // handle decryption failure
        $row['api_key_plain'] = null;
        $row['api_secret_plain'] = null;
        $row['passphrase_plain'] = null;
    }
    return $row;
}
