<?php
header('Content-Type: application/json');

// ===== 🔐 CONFIGURATION =====
define('AUTH_TOKEN', 'your_secure_token_here');
define('ALERT_THRESHOLD', 5000); // Profit in NGN
define('ALERT_EMAIL', 'you@example.com');
define('PROFIT_LOG_FILE', __DIR__ . '/profits.json');
define('ALERT_LOG_FILE', __DIR__ . '/alerts.log');

// ===== 🔔 TELEGRAM CONFIG =====
define('TELEGRAM_BOT_TOKEN', 'YOUR_TELEGRAM_BOT_TOKEN');
define('TELEGRAM_CHAT_ID', 'YOUR_TELEGRAM_CHAT_ID');

// ===== 🔐 TOKEN VALIDATION =====
$headers = getallheaders();
$token = $headers['Authorization'] ?? $_GET['token'] ?? '';
if (trim($token) !== AUTH_TOKEN) {
    http_response_code(401);
    echo json_encode(['status' => 'unauthorized']);
    exit;
}

// ===== 🟢 INPUT HANDLING =====
$input = $_SERVER['REQUEST_METHOD'] === 'POST' 
    ? json_decode(file_get_contents("php://input"), true)
    : $_GET;

if (
    !isset($input['buy']) || !isset($input['sell']) ||
    !isset($input['exchange'])
) {
    echo json_encode(['status' => 'error', 'message' => 'Missing fields']);
    exit;
}

$buy      = round(floatval($input['buy']), 2);
$sell     = round(floatval($input['sell']), 2);
$exchange = htmlspecialchars(trim($input['exchange']));
$time     = date('Y-m-d H:i:s');
$profit   = round($sell - $buy, 2);

// ===== 💾 SAVE TRADE ENTRY =====
$entry = [
    'time'     => $time,
    'profit'   => $profit,
    'buy'      => $buy,
    'sell'     => $sell,
    'exchange' => $exchange
];

$profits = file_exists(PROFIT_LOG_FILE) ? json_decode(file_get_contents(PROFIT_LOG_FILE), true) : [];
$profits[] = $entry;
$profits = array_slice($profits, -100); // Keep last 100
file_put_contents(PROFIT_LOG_FILE, json_encode($profits, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES), LOCK_EX);

// ===== 🚨 ALERT ON THRESHOLD EXCEED =====
if ($profit >= ALERT_THRESHOLD) {
    $alertText = "🚨 *Profit Alert!*\n"
               . "*Exchange:* {$exchange}\n"
               . "*Buy:* ₦" . number_format($buy, 2) . "\n"
               . "*Sell:* ₦" . number_format($sell, 2) . "\n"
               . "*Profit:* ₦" . number_format($profit, 2) . "\n"
               . "*Time:* {$time}";

    // 📧 Email Alert
    @mail(ALERT_EMAIL, "Profit Alert: ₦{$profit}", strip_tags($alertText), "From: profitbot@yourdomain.com");

    // 🗃️ Log Alert
    $logLine = "[" . date("Y-m-d H:i:s") . "] " . str_replace("\n", " | ", strip_tags($alertText)) . "\n";
    file_put_contents(ALERT_LOG_FILE, $logLine, FILE_APPEND | LOCK_EX);

    // 💬 Telegram Alert
    $telegramUrl = "https://api.telegram.org/bot" . TELEGRAM_BOT_TOKEN . "/sendMessage";
    $telegramPayload = [
        'chat_id'    => TELEGRAM_CHAT_ID,
        'text'       => $alertText,
        'parse_mode' => 'Markdown'
    ];
    @file_get_contents($telegramUrl . '?' . http_build_query($telegramPayload));
}

// ===== ✅ RESPONSE =====
echo json_encode([
    'status' => 'ok',
    'data'   => $entry
]);
