<?php
// ======= CONFIG =======
$logFile = __DIR__ . "/trades.json";
define('ALERT_THRESHOLD', 5000); // NGN
define('ENABLE_TELEGRAM', true);

// Telegram Config
define('TELEGRAM_BOT_TOKEN', 'YOUR_BOT_TOKEN'); // Replace
define('TELEGRAM_CHAT_ID', 'YOUR_CHAT_ID');     // Replace

// ======= HELPERS =======
function fetch_binance_rate($type) {
    $payload = [
        "asset" => "USDT",
        "fiat" => "NGN",
        "tradeType" => $type,
        "page" => 1,
        "rows" => 5
    ];
    $ctx = stream_context_create([
        'http' => [
            'header'  => "Content-Type: application/json",
            'method'  => 'POST',
            'timeout' => 10,
            'content' => json_encode($payload)
        ]
    ]);
    $res = @file_get_contents("https://p2p.binance.com/bapi/c2c/v2/friendly/c2c/adv/search", false, $ctx);
    if (!$res) return null;
    $json = json_decode($res, true);
    if (!isset($json['data'])) return null;
    return array_map(fn($item) => floatval($item['adv']['price']), $json['data']);
}

function fetch_kucoin_rate() {
    $res = @file_get_contents("https://www.kucoin.com/_api/c2c/otc/quote/USDT-NGN");
    if (!$res) return null;
    $json = json_decode($res, true);
    if (!isset($json['data'])) return null;
    return [
        'buy' => floatval($json['data']['buyPrice']),
        'sell' => floatval($json['data']['sellPrice'])
    ];
}

function fetch_bybit_rate() {
    $res = @file_get_contents("https://api2.bybit.com/fiat/otc/item/online?userId=&tokenId=USDT&currencyId=NGN&payment=&side=1&size=5&page=1");
    if (!$res) return null;
    $json = json_decode($res, true);
    if (!isset($json['result']['items'])) return null;
    $buyPrices = array_map(fn($item) => floatval($item['price']), $json['result']['items']);

    $res2 = @file_get_contents("https://api2.bybit.com/fiat/otc/item/online?userId=&tokenId=USDT&currencyId=NGN&payment=&side=2&size=5&page=1");
    $json2 = json_decode($res2, true);
    $sellPrices = isset($json2['result']['items']) ? array_map(fn($item) => floatval($item['price']), $json2['result']['items']) : [];

    return [
        'buy' => min($buyPrices),
        'sell' => max($sellPrices)
    ];
}

function fetch_gateio_rate() {
    $res = @file_get_contents("https://www.gate.io/json_svr/query_push_list?type=otc&coin=usdt&curr=ngn");
    if (!$res) return null;
    $json = json_decode($res, true);
    if (!isset($json['bids'], $json['asks'])) return null;
    $buyPrices = array_map(fn($b) => floatval($b[0]), $json['bids']);
    $sellPrices = array_map(fn($s) => floatval($s[0]), $json['asks']);
    return [
        'buy' => min($buyPrices),
        'sell' => max($sellPrices)
    ];
}

// ======= AGGREGATE =======
$exchanges = [];

$binanceBuy  = fetch_binance_rate("BUY");
$binanceSell = fetch_binance_rate("SELL");
if ($binanceBuy && $binanceSell) {
    $exchanges[] = [
        'exchange' => 'Binance P2P',
        'buy' => min($binanceBuy),
        'sell' => max($binanceSell)
    ];
}

$kucoin = fetch_kucoin_rate();
if ($kucoin) {
    $exchanges[] = ['exchange' => 'KuCoin P2P', 'buy' => $kucoin['buy'], 'sell' => $kucoin['sell']];
}

$bybit = fetch_bybit_rate();
if ($bybit) {
    $exchanges[] = ['exchange' => 'Bybit P2P', 'buy' => $bybit['buy'], 'sell' => $bybit['sell']];
}

$gate = fetch_gateio_rate();
if ($gate) {
    $exchanges[] = ['exchange' => 'Gate.io P2P', 'buy' => $gate['buy'], 'sell' => $gate['sell']];
}

// ======= CALCULATE MAX PROFIT OPPORTUNITY =======
$bestOpportunity = null;
$maxProfit = 0;

foreach ($exchanges as $buyEx) {
    foreach ($exchanges as $sellEx) {
        if ($buyEx['exchange'] === $sellEx['exchange']) continue;
        $profit = ($sellEx['sell'] - $buyEx['buy']) * 100;
        if ($profit > $maxProfit) {
            $maxProfit = $profit;
            $bestOpportunity = [
                'buy_exchange' => $buyEx['exchange'],
                'sell_exchange' => $sellEx['exchange'],
                'buy_rate' => $buyEx['buy'],
                'sell_rate' => $sellEx['sell'],
                'profit' => round($profit, 2),
                'timestamp' => date("Y-m-d H:i:s")
            ];
        }
    }
}

// ======= OUTPUT / ALERT =======
if (!$bestOpportunity) {
    echo json_encode(['success' => false, 'error' => 'No valid arbitrage opportunity']);
    exit;
}

if ($bestOpportunity['profit'] >= ALERT_THRESHOLD) {
    // Log
    $log = file_exists($logFile) ? json_decode(file_get_contents($logFile), true) : [];
    $log[] = $bestOpportunity;
    $log = array_slice($log, -100);
    file_put_contents($logFile, json_encode($log, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX);

    // Telegram
    if (ENABLE_TELEGRAM) {
        $msg = "🚨 *Arbitrage Opportunity!*\n"
             . "*Buy from:* " . $bestOpportunity['buy_exchange'] . " at ₦" . number_format($bestOpportunity['buy_rate'], 2) . "\n"
             . "*Sell on:* " . $bestOpportunity['sell_exchange'] . " at ₦" . number_format($bestOpportunity['sell_rate'], 2) . "\n"
             . "*Profit (₦/100 USDT):* " . number_format($bestOpportunity['profit'], 2) . "\n"
             . "*Time:* " . $bestOpportunity['timestamp'];

        $url = "https://api.telegram.org/bot" . TELEGRAM_BOT_TOKEN . "/sendMessage?" . http_build_query([
            'chat_id' => TELEGRAM_CHAT_ID,
            'text' => $msg,
            'parse_mode' => 'Markdown'
        ]);

        @file_get_contents($url);
    }
}

echo json_encode(['success' => true] + $bestOpportunity);
