<?php
header('Content-Type: application/json');

// ===== 🔐 CONFIGURATION =====
define('AUTH_TOKEN', 'your_secure_token_here'); // Change this!
define('ALERT_THRESHOLD', 5000);                // Profit threshold in NGN
define('ALERT_EMAIL', 'you@example.com');
define('LOG_DIR', __DIR__ . '/logs'); // Directory for per-exchange logs
define('PROFIT_LOG_FILE', __DIR__ . '/profits.json');

// ===== 🔔 TELEGRAM CONFIG =====
define('TELEGRAM_BOT_TOKEN', 'YOUR_TELEGRAM_BOT_TOKEN'); // Change this!
define('TELEGRAM_CHAT_ID', 'YOUR_TELEGRAM_CHAT_ID');     // Change this!

// ===== 🔐 TOKEN VALIDATION =====
$headers = getallheaders();
$token = $headers['Authorization'] ?? $_GET['token'] ?? '';
if (trim($token) !== AUTH_TOKEN) {
    http_response_code(401);
    echo json_encode(['status' => 'unauthorized']);
    exit;
}

// ===== 🟢 INPUT =====
$input = json_decode(file_get_contents("php://input"), true);
if (!$input || !isset($input['profit'], $input['time'], $input['buy'], $input['sell'], $input['exchange'])) {
    echo json_encode(['status' => 'error', 'message' => 'Missing required fields']);
    exit;
}

// ===== 📊 SANITIZE INPUT =====
$profit   = round(floatval($input['profit']), 2);
$time     = htmlspecialchars(trim($input['time']));
$buy      = round(floatval($input['buy']), 2);
$sell     = round(floatval($input['sell']), 2);
$exchange = htmlspecialchars(trim($input['exchange']));
$exchangeSafe = preg_replace('/[^a-zA-Z0-9_]/', '', strtolower($exchange));

// ===== 💾 SAVE TO PROFITS LOG =====
$entry = [
    'time'     => $time,
    'profit'   => $profit,
    'buy'      => $buy,
    'sell'     => $sell,
    'exchange' => $exchange
];

$profits = file_exists(PROFIT_LOG_FILE) ? json_decode(file_get_contents(PROFIT_LOG_FILE), true) : [];
$profits[] = $entry;
$profits = array_slice($profits, -100); // Keep last 100 entries
file_put_contents(PROFIT_LOG_FILE, json_encode($profits, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES), LOCK_EX);

// ===== 📁 Ensure log directory exists =====
if (!file_exists(LOG_DIR)) {
    mkdir(LOG_DIR, 0777, true);
}

// ===== 🗃️ Per-exchange Logging (with rotation) =====
$logFile = LOG_DIR . "/{$exchangeSafe}_alerts.log";
$archiveFile = LOG_DIR . "/archive_" . date('Ymd') . "_{$exchangeSafe}.log";
if (file_exists($logFile) && filesize($logFile) > 500 * 1024) {
    rename($logFile, $archiveFile);
}

// ===== 🚨 IF ABOVE THRESHOLD, ALERT =====
if ($profit >= ALERT_THRESHOLD) {
    $alertText = "🚨 *Profit Opportunity Detected!*
"
               . "*Exchange:* {$exchange}
"
               . "*Buy Rate:* ₦" . number_format($buy, 2) . "
"
               . "*Sell Rate:* ₦" . number_format($sell, 2) . "
"
               . "*Profit (₦/100 USDT):* " . number_format($profit, 2) . "
"
               . "*Time:* {$time}";

    // ===== 📧 EMAIL NOTIFICATION =====
    @mail(ALERT_EMAIL, "Profit Alert: ₦$profit from $exchange", strip_tags($alertText), "From: alerts@yourdomain.com");

    // ===== 🗃️ LOG TO FILE (exchange-specific) =====
    $logLine = "[" . date("Y-m-d H:i:s") . "] " . str_replace("\n", " | ", strip_tags($alertText)) . "\n";
    file_put_contents($logFile, $logLine, FILE_APPEND | LOCK_EX);

    // ===== 💬 TELEGRAM ALERT =====
    $telegramUrl = "https://api.telegram.org/bot" . TELEGRAM_BOT_TOKEN . "/sendMessage";
    $telegramPayload = [
        'chat_id'    => TELEGRAM_CHAT_ID,
        'text'       => $alertText,
        'parse_mode' => 'Markdown'
    ];
    @file_get_contents($telegramUrl . '?' . http_build_query($telegramPayload));
}

// ===== ✅ RESPONSE =====
echo json_encode([
    'status' => 'ok',
    'saved'  => $entry
]);
?>
