<?php
// admin_api/exchange_test.php
// Tests decrypted API credentials for Binance, KuCoin, Gate.io (signed tests).
// Requires: auth.php (for Authorization bearer), includes/exchange_store.php (get_exchange_credentials)

require_once __DIR__ . '/auth.php';
require_once __DIR__ . '/../includes/exchange_store.php';
global $pdo;

header('Content-Type: application/json; charset=utf-8');

$body = json_decode(file_get_contents('php://input'), true) ?: [];
$exchange = trim($body['exchange'] ?? '');
if (!$exchange) {
    http_response_code(400);
    echo json_encode(['ok' => false, 'code' => 400, 'message' => 'Missing "exchange" in request']);
    exit;
}

// Fetch credentials (reveal=true because we need plaintext to test)
// Note: Only allow reveal for authenticated admins; auth.php should enforce this.
$row = $pdo ? get_exchange_credentials($pdo, $exchange, true) : null;
if (!$row) {
    // fallback to file data
    $f = __DIR__ . '/data/exchange_credentials.json';
    if (file_exists($f)) {
        $all = json_decode(file_get_contents($f), true);
        foreach ($all as $r) {
            if (strcasecmp($r['exchange'], $exchange) === 0) { $row = $r; break; }
        }
    }
}
if (!$row) {
    http_response_code(404);
    echo json_encode(['ok' => false, 'code' => 404, 'message' => 'Credentials not found for exchange']);
    exit;
}

// pull plaintext creds (may be null)
$apiKey = $row['api_key_plain'] ?? null;
$apiSecret = $row['api_secret_plain'] ?? null;
$passphrase = $row['passphrase_plain'] ?? null;

// helper: simple curl with timeout
function curl_request($method, $url, $headers = [], $body = null, $timeout = 8) {
    $ch = curl_init();
    $opts = [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_CUSTOMREQUEST => $method,
        CURLOPT_TIMEOUT => $timeout,
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_FAILONERROR => false, // we'll capture HTTP code ourselves
    ];
    if ($body !== null) {
        if (is_array($body) || is_object($body)) {
            $body = json_encode($body);
            $headers[] = 'Content-Type: application/json';
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        }
        $opts[CURLOPT_POSTFIELDS] = $body;
    }
    curl_setopt_array($ch, $opts);
    $resp = curl_exec($ch);
    $errno = curl_errno($ch);
    $err = curl_error($ch);
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    return ['errno'=>$errno, 'error'=>$err, 'code'=>$code, 'body'=>$resp];
}

// Prepare result skeleton
$result = ['ok'=>false, 'code'=>null, 'message'=>'', 'details'=>null];

try {
    switch (strtolower($exchange)) {
        case 'binance':
            // Binance signed endpoint test (account) - requires API key + secret
            // NOTE: This uses the signed /api/v3/account endpoint. If the key is IP-restricted it may fail.
            if (empty($apiKey) || empty($apiSecret)) {
                $result['message'] = 'API key or secret missing';
                break;
            }
            $base = 'https://api.binance.com';
            $path = '/api/v3/account';
            $ts = round(microtime(true) * 1000);
            $query = 'timestamp=' . $ts . '&recvWindow=5000';
            $signature = hash_hmac('sha256', $query, $apiSecret);
            $url = $base . $path . '?' . $query . '&signature=' . $signature;
            $headers = ['X-MBX-APIKEY: ' . $apiKey, 'User-Agent: Arbitrage-Test/1.0'];
            $r = curl_request('GET', $url, $headers, null, 8);
            if ($r['errno']) {
                $result['message'] = 'cURL error: ' . $r['error'];
                $result['code'] = 520;
            } elseif ($r['code'] >= 200 && $r['code'] < 300) {
                $result['ok'] = true;
                $result['code'] = $r['code'];
                $result['message'] = 'Binance account endpoint reachable. Key likely valid.';
                // Avoid returning sensitive payload; offer limited info
                $json = json_decode($r['body'], true);
                $result['details'] = [
                    'balances_count' => isset($json['balances']) ? count($json['balances']) : null,
                    'makerCommission' => $json['makerCommission'] ?? null
                ];
            } else {
                // parse body for API error if present
                $body = $r['body'] ? json_decode($r['body'], true) : null;
                $msg = $body['msg'] ?? $body['message'] ?? 'HTTP ' . $r['code'];
                $result['message'] = 'Binance API error: ' . $msg;
                $result['code'] = $r['code'];
                $result['details'] = $body;
            }
            break;

        case 'kucoin':
            // KuCoin signed endpoint test (/api/v1/accounts) uses HMAC-SHA256 over (timestamp + method + requestPath + body)
            // Docs: https://docs.kucoin.com/
            if (empty($apiKey) || empty($apiSecret) || empty($passphrase)) {
                $result['message'] = 'API key, secret, or passphrase missing (KuCoin requires passphrase)';
                break;
            }
            $base = 'https://api.kucoin.com';
            $path = '/api/v1/accounts'; // lightweight accounts listing
            $method = 'GET';
            $now = (string) round(microtime(true) * 1000);
            $what = $now . $method . $path; // no body for GET
            $signature = base64_encode(hash_hmac('sha256', $what, $apiSecret, true));
            $headers = [
                'KC-API-KEY: ' . $apiKey,
                'KC-API-SIGN: ' . $signature,
                'KC-API-TIMESTAMP: ' . $now,
                'KC-API-PASSPHRASE: ' . base64_encode($passphrase),
                'KC-API-KEY-VERSION: 2',
                'User-Agent: Arbitrage-Test/1.0'
            ];
            $url = $base . $path;
            $r = curl_request($method, $url, $headers, null, 8);
            if ($r['errno']) {
                $result['message'] = 'cURL error: ' . $r['error'];
                $result['code'] = 520;
            } elseif ($r['code'] >= 200 && $r['code'] < 300) {
                $result['ok'] = true;
                $result['code'] = $r['code'];
                $result['message'] = 'KuCoin accounts endpoint reachable. Key likely valid.';
                $json = json_decode($r['body'], true);
                $result['details'] = [
                    'data_count' => isset($json['data']) ? (is_array($json['data']) ? count($json['data']) : null) : null
                ];
            } else {
                $body = $r['body'] ? json_decode($r['body'], true) : null;
                $errMsg = $body['msg'] ?? $body['message'] ?? 'HTTP ' . $r['code'];
                $result['message'] = 'KuCoin API error: ' . $errMsg;
                $result['code'] = $r['code'];
                $result['details'] = $body;
            }
            break;

        case 'gate.io':
        case 'gateio':
            // Gate.io API v4 signed endpoint test (wallet accounts)
            // Docs: https://www.gate.io/docs/apiv4/en/index.html
            if (empty($apiKey) || empty($apiSecret)) {
                $result['message'] = 'API key or secret missing';
                break;
            }
            $base = 'https://api.gateio.ws';
            $path = '/api/v4/wallet/coin_accounts';
            $method = 'GET';
            $ts = gmdate('Y-m-d\TH:i:s\Z');
            // Gate.io signature: HMAC_SHA512(api_secret, time + method + request_path + body)
            $what = $ts . $method . $path; // no body for GET
            $sign = hash_hmac('sha512', $what, $apiSecret);
            $headers = [
                'KEY: ' . $apiKey,
                'Timestamp: ' . $ts,
                'SIGN: ' . $sign,
                'User-Agent: Arbitrage-Test/1.0'
            ];
            $url = $base . $path;
            $r = curl_request($method, $url, $headers, null, 8);
            if ($r['errno']) {
                $result['message'] = 'cURL error: ' . $r['error'];
                $result['code'] = 520;
            } elseif ($r['code'] >= 200 && $r['code'] < 300) {
                $result['ok'] = true;
                $result['code'] = $r['code'];
                $result['message'] = 'Gate.io wallet endpoint reachable. Key likely valid.';
                $json = json_decode($r['body'], true);
                $result['details'] = [
                    'accounts_count' => is_array($json) ? count($json) : null
                ];
            } else {
                $body = $r['body'] ? json_decode($r['body'], true) : null;
                $errMsg = $body['message'] ?? (is_array($body) ? json_encode($body) : ($r['body'] ?: 'HTTP ' . $r['code']));
                $result['message'] = 'Gate.io API error: ' . $errMsg;
                $result['code'] = $r['code'];
                $result['details'] = $body;
            }
            break;

        default:
            $result['message'] = 'Test not implemented for this exchange. You can extend admin_api/exchange_test.php.';
            $result['code'] = 501;
            break;
    }
} catch (Exception $e) {
    $result['message'] = 'Exception: ' . $e->getMessage();
    $result['code'] = 500;
} finally {
    // wipe sensitive variables from memory
    if (isset($apiKey)) { unset($apiKey); }
    if (isset($apiSecret)) { unset($apiSecret); }
    if (isset($passphrase)) { unset($passphrase); }
}

// Output (never include secrets)
echo json_encode($result, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
exit;
